<?php

/**
 * This is the model class for table "Klasa".
 *
 * The followings are the available columns in table 'Klasa':
 * @property double $id
 * @property string $nazwa
 * @property double $idSzkoly
 * @property string $data
 * @property int $czyTestowa
 */
class Klasa extends CActiveRecord
{
	public $searchSchoolIdFilter; //used by search function to find classes belonging only to this school
	public $teacherName; //finding by teacher
	public $teacherId;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Klasa the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Klasa';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('teacherId, czasRozpoczeciaRejestracji', 'safe'),
			array('idSzkoly,nazwa', 'required'),
			array('idSzkoly', 'numerical'),
			array('nazwa', 'length', 'max'=>64),

			array('id, nazwa, idSzkoly, teacherName', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'admini'=>array(self::MANY_MANY, 'Admin', 'AdminKlasy(idKlasy, idAdmina)'),
			'szkola'=>array(self::BELONGS_TO, 'Szkola', 'idSzkoly'),
			'turas'=>array(self::HAS_MANY, 'Tura', 'idKlasy'),
			'uzytkowniks'=>array(self::HAS_MANY, 'Uzytkownik', 'idKlasy'),
			'firmy'=>array(self::HAS_MANY, 'Firma', 'idKlasy'),
			'nieaktywneFirmy'=>array(self::HAS_MANY, 'Firma', 'idKlasy', 'on'=>'"nieaktywneFirmy"."status"='.Firma::COMPANY_STATUS_INACTIVE),
			'aktywneFirmy'=>array(self::HAS_MANY, 'Firma', 'idKlasy', 'on'=>'"aktywneFirmy"."status"='.Firma::COMPANY_STATUS_ACTIVE),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Klasa', 'ID'),
			'nazwa'=>Yii::t('Klasa', 'Nazwa'),
			'idSzkoly'=>Yii::t('Klasa', 'Szkoła'),
			'teacherName'=>Yii::t('Klasa', 'Nauczyciel'),
		);
	}

	public static function getCompanies($classId)
	{
		$sql = 'SELECT t."id", t."nazwa"
			FROM "Firma" t
			WHERE t."idKlasy" = '.$classId.'
			ORDER BY t."nazwa"';

		return Yii::app()->db->createCommand($sql)->queryAll();
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;

		$criteria->compare(Yii::app()->db->quoteColumnName('id'), $this->id);
		$criteria->compare('LOWER("t"."nazwa")', strtolower($this->nazwa), true);
		$criteria->compare('"t"."czyTestowa"', $this->czyTestowa, true);
		$criteria->together = true;
		$criteria->compare('LOWER("admini"."imie"||\' \'||"admini"."nazwisko")', strtolower($this->teacherName), true);

		if(!empty($this->searchSchoolIdFilter)) {
			$criteria->compare(Yii::app()->db->quoteTableName('t').'.'.Yii::app()->db->quoteColumnName('idSzkoly'), $this->searchSchoolIdFilter);
		}
		else
			$criteria->compare(Yii::app()->db->quoteTableName('t').'.'.Yii::app()->db->quoteColumnName('idSzkoly'), $this->idSzkoly);

		$criteria->with = array('szkola','admini');

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
			'sort'=>array(
				'defaultOrder'=>Yii::app()->db->quoteTableName('t').'.'.Yii::app()->db->quoteColumnName('nazwa').' ASC',
			),
		));
	}

	public function getDocuments()
	{
		$sql = 'FROM "Uzytkownik" t
			LEFT JOIN "FormularzUzytkownika" t2 ON t."id" = t2."idUzytkownika"
			LEFT JOIN "Firma" t3 ON t."idFirmy" = t3."id"
			LEFT JOIN "Formularz" t5 ON t2."idFormularza" = t5."id"
			WHERE t."idKlasy" = :klasaId AND t2."status" IN(:s1, :s2, :s3)
			ORDER BY t2."dataModyfikacji"';

		$params = array(
			':klasaId' => $this->id,
			':s1' => FormularzUzytkownika::STATUS_FILLED,
			':s2' => FormularzUzytkownika::STATUS_ACCEPTED,
			':s3' => FormularzUzytkownika::STATUS_REFUSED,
		);

		$totalItemsCount = Yii::app()->db->createCommand('SELECT count(*) ' . $sql)->queryScalar($params);

		$sql = 'SELECT t."imie", t."nazwisko", t2."id" as "id", t3."id" as "idFirmy", t3."nazwa" as "nazwaFirmy",
			t5."nazwa" as "nazwaFormularza", t2."dataModyfikacji", t2."status" ' . $sql;

		return new CSqlDataProvider($sql, array(
			'totalItemCount' => $totalItemsCount,
			'params' => $params,
			'sort' => array(
				'attributes' => array(
					'"nazwaFirmy"', '"dataModyfikacji"'
				),
			),
			'pagination' => array(
				'pageSize' => 10,
			)
		));
	}

	public function getRegistrationTime()
	{
		$sql = 'SELECT to_char("czasRozpoczeciaRejestracji",\'YYYY-MM-DD HH24:MI:SS\')
			FROM "Klasa"
			WHERE "id"='.Yii::app()->db->quoteValue($this->id).'';

		return Yii::app()->db->createCommand($sql)->queryScalar();
	}

	public static function registrationAvailable($classId)
	{
		$sql = 'SELECT CASE
				WHEN t."czasRozpoczeciaRejestracji"+TO_NUMBER(t2."wartosc")/1440 > SYSDATE AND SYSDATE > t."czasRozpoczeciaRejestracji"
					THEN 1
				ELSE
					0 END as "registrationAvailable"
			FROM "Klasa" t
			LEFT JOIN "ParametryGry" t2 ON t2."nazwa" = \'CZAS_NA_REJESTRACJE\'
			WHERE t."id"='.Yii::app()->db->quoteValue($classId);

		return Yii::app()->db->createCommand($sql)->queryScalar();
	}

	public function getClassesWithAvailableRegistration()
	{
		$sql = 'SELECT t."miasto" as "miasto", t."nazwa" as "szkola", t."id" as "idSzkoly", t2."nazwa" as "klasa", t2."id" as "idKlasy"
			FROM "Szkola" t
			JOIN "Klasa" t2 ON (t2."idSzkoly" = t."id")
			LEFT JOIN "ParametryGry" t3 ON (t3."nazwa" = \'CZAS_NA_REJESTRACJE\')
			WHERE t2."czasRozpoczeciaRejestracji"+TO_NUMBER(t3."wartosc")/1440 > SYSDATE AND SYSDATE > t2."czasRozpoczeciaRejestracji"';

		$classes = Yii::app()->db->createCommand($sql)->queryAll();

		$classesResult = array();

		foreach($classes as $class)
		{
			$classesResult[$class['miasto']][$class['idSzkoly']]['klasy'][$class['idKlasy']] = $class['klasa'];
			$classesResult[$class['miasto']][$class['idSzkoly']]['nazwaSzkoly'] = $class['szkola'];
		}

		return $classesResult;
	}

	public function getNumberOfRegistrationsAvailable()
	{
		$sql = 'SELECT COUNT(*) as "numberOfRegistrations"
			FROM "Szkola" t
			JOIN "Klasa" t2 ON (t2."idSzkoly" = t."id")
			LEFT JOIN "ParametryGry" t3 ON (t3."nazwa" = \'CZAS_NA_REJESTRACJE\')
			WHERE t2."czasRozpoczeciaRejestracji"+TO_NUMBER(t3."wartosc")/1440 > SYSDATE AND SYSDATE > t2."czasRozpoczeciaRejestracji"';

		return Yii::app()->db->createCommand($sql)->queryScalar();
	}

	public function getAllStatistics($adminId)
	{
		$currentTestClassId = self::getCurrentTestClassId($adminId);

		$sql = 'SELECT "nazwa" as "klasa", "punkty", "idKlasy" as "id"
			FROM "PunktyKlasy"
			WHERE "idKlasy" IN (SELECT t1."idKlasy" FROM "AdminKlasy" t1
						LEFT JOIN "Klasa" t2 ON t2."id" = t1."idKlasy"
						WHERE t1."idAdmina"='.Yii::app()->db->quoteValue($adminId).' AND (t2."czyTestowa" != 1 OR (t2."czyTestowa"=1 AND t2."id"='.$currentTestClassId.')))
			ORDER BY "punkty" DESC';

			return new CSqlDataProvider($sql,array(
				'pagination'=>false
			));
	}

	public function getAllStatisticsBySchool($schoolId)
	{
		$sql = 'SELECT *
			FROM "PunktyKlasy" t
			LEFT JOIN "Klasa" t2 ON t."idKlasy" = t2."id"
			LEFT JOIN "AdminKlasy" t3 ON t3."idKlasy" = t2."id"
			LEFT JOIN "Admin" t4 ON t4."id" = t3."idAdmina"
			WHERE t2."idSzkoly" = '.Yii::app()->db->quoteValue($schoolId).' AND t2."czyTestowa" = 0
			ORDER BY t."punkty" DESC';

		return new CSqlDataProvider($sql,array(
			'keyField'=>'idKlasy',
			'pagination'=>false
		));
	}

	public function getOrders($classId)
	{
		$sql = 'select p."id", k."nazwa", d."nazwa" "dziedzina", p."poziomTrudnosci" "poziom" from "Przetarg" p
			left join "Kontrahent" k on p."idKontrahenta" = k."id"
			left join "Dziedzina" d on d."id" = k."idDziedziny"
			left join "PrzetargGrupa" pg on pg."idPrzetargu" = p."id" and pg."status" = :status
			where "idKlasy" = :classId
			group by p."id", k."nazwa", d."nazwa", p."poziomTrudnosci"
			having sum(nvl2(pg."id", 1, 0)) > 0';

		return new CSqlDataProvider($sql,array(
			'pagination'=>false,
			'params' => array(
				':status' => PrzetargGrupa::TENDER_STATUS_ACCEPTATION_REQUEST,
				':classId' => $classId
			)
		));
	}

	public static function getTeacherClass($classId)
	{
		return Klasa::model()->find(array(
			'join' => 'LEFT JOIN "AdminKlasy" "ak" ON "ak"."idKlasy" = "t"."id"',
			'condition' => '"t"."id" = :classId AND "ak"."idAdmina" = :teacherId',
			'params' => array(
				':classId' => $classId,
				':teacherId' => Yii::app()->user->id
			)
		));
	}

	public static function createTestClassIfNotExists()
	{
		$sql = 'SELECT COUNT(*) FROM "Klasa" "k"
			LEFT JOIN "AdminKlasy" "ak" ON "ak"."idKlasy" = "k"."id"
			WHERE "ak"."idAdmina" = '.Yii::app()->user->id.' AND "k"."czyTestowa" = 1';

		$numberOfTestClasses = Yii::app()->db->createCommand($sql)->queryScalar();

		if($numberOfTestClasses == 0)
			self::createTeachersTestClass();
	}

	public static function createTeachersTestClass()
	{
		$transaction = Yii::app()->db->beginTransaction();

		try
		{
			$teacher = Admin::model()->findByPk(Yii::app()->user->id);
			$class = new Klasa;
			$class->nazwa = 'Klasa testowa';
			$class->idSzkoly = $teacher->idSzkoly;
			$class->czyTestowa = true;
			$class->save(false);

			$classAdmin = new AdminKlasy;
			$classAdmin->idKlasy = $class->id;
			$classAdmin->idAdmina = $teacher->id;
			$classAdmin->save(false);

			$company = new Firma();
			$company->czasUtworzenia = new CDbExpression('SYSDATE');
			$company->nazwa = "Testowa Firma";
			$company->idKlasy = $class->id;
			$company->status = Firma::COMPANY_STATUS_ACTIVE;
			$company->save(false);

			$fields = Dziedzina::model()->findAll();

			foreach ($fields as $field)
			{
				$companyField = new DziedzinaFirmy();
				$companyField->idFirmy = intval($company->id);
				$companyField->idDziedziny = intval($field->id);
				$companyField->punkty = intval(0);
				$companyField->save(false);
			}

			$testUserLoginData = $class->getTestUserLoginData();
			$testUser = new Uzytkownik;
			$testUser->haslo = hash('sha1', 'testowy_'.$teacher->login);
			$testUser->imie = 'Imię';
			$testUser->nazwisko = 'Nazwisko';
			$testUser->email = $testUserLoginData['email'];
			$testUser->haslo = hash('sha1', $testUserLoginData['password']);
			$testUser->dataRejestracji = new CDbExpression('SYSDATE');
			$testUser->status = Uzytkownik::USER_STATUS_ACTIVE;
			$testUser->plec = 'm';
			$testUser->idKlasy = $class->id;
			$testUser->idFirmy = $company->id;
			$testUser->save(false);

			foreach($fields as $field)
			{
				$companyField = new DziedzinaUzytkownika();
				$companyField->idUzytkownika = $testUser->id;
				$companyField->idDziedziny = $field->id;
				$companyField->punkty = 0;
				$companyField->save(false);
			}

			$transaction->commit();
		}
		catch (Exception $e)
		{
			$transaction->rollback();
			throw $e;
		}
	}

	public function getTestUserLoginData()
	{
		return array('email'=>'testowy_'.$this->id.'@kreatywnie.lublin.eu','password'=>'haslo_'.$this->id.'_'.$this->admini[0]->id);
	}

	public function getCurrentTestClassId($adminId)
	{
		$selectedClass = Klasa::model()->with("admini")->find(array(
			'select'=>'"t"."id"',
			'condition'=>'"admini"."id" = :adminid AND "t"."czyTestowa" = 1',
			'order'=>'"t"."id" DESC',
			'params'=>array(':adminid'=>$adminId)
		));
		if(empty($selectedClass))
		{
			return null;
		}
		else
		{
			return $selectedClass->id;
		}
	}
}
